import asyncio
import os
import json
import shutil
import re

from datetime import datetime

import numpy as np
import matplotlib.pyplot as plt

from py_pli.pylib import VUnits
from py_pli.pylib import GlobalVar
from py_pli.pylib import send_gc_event

from config_enum import hal_enum as hal_config

from virtualunits.HAL import HAL
from virtualunits.vu_node_application import VUNodeApplication
from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.VirtualTemperatureUnit import VirtualTemperatureUnit
from virtualunits.VirtualFanUnit import VUFanControl
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel
from virtualunits.meas_seq_generator import IntegratorMode
from virtualunits.meas_seq_generator import AnalogControlMode

from predefined_tasks.common.helper import send_to_gc
from predefined_tasks.common.node_io import FMBAnalogOutput
from predefined_tasks.common.node_io import EEFAnalogInput
from predefined_tasks.common.node_io import EEFAnalogOutput
from predefined_tasks.common.node_io import EEFDigitalOutput

from urpc.nodefunctions import NodeFunctions
from urpc.measurementfunctions import MeasurementFunctions

from urpc_enum.measurementparameter import MeasurementParameter

from fleming.rbartz.flash_lum_test import flash_lum_get_led_power

hal_unit: HAL = VUnits.instance.hal
eef_unit: VUNodeApplication = hal_unit.nodes['EEFNode']
fmb_unit: VUNodeApplication = hal_unit.nodes['Mainboard']
meas_unit: VUMeasurementUnit = hal_unit.measurementUnit
pmt1_cooling: VirtualTemperatureUnit = hal_unit.pmt_ch1_Cooling
pmt2_cooling: VirtualTemperatureUnit = hal_unit.pmt_ch2_Cooling
uslum_fan: VUFanControl = hal_unit.usLum_Fan

eef_endpoint: NodeFunctions = eef_unit.endpoint
fmb_endpoint: NodeFunctions = fmb_unit.endpoint
meas_endpoint: MeasurementFunctions = meas_unit.endpoint

report_path = hal_unit.configManager.get_config(hal_config.Application.GCReportPath)
images_path = os.path.join(report_path, 'pmt_adjust_images')
archive_path = os.path.join(report_path, 'pmt_adjust_archive')
calibration_path = os.path.join(report_path, 'pmt_adjust_calibration')

os.makedirs(report_path, exist_ok=True)
os.makedirs(images_path, exist_ok=True)
os.makedirs(archive_path, exist_ok=True)
os.makedirs(calibration_path, exist_ok=True)

config_file = os.path.join(calibration_path, 'pmt_adjust_config.json')
if os.path.isfile(config_file):
    with open(config_file, 'r') as file:
        config = json.load(file)
else:
    config = {}

led_dimmed = {'pmt1':'fmb_led2_green', 'pmt2':'fmb_led4_green', 'pmt3':'fmb_led2_green'}
led_bright = {'pmt1':'fmb_led1_green', 'pmt2':'fmb_led3_green', 'pmt3':'fmb_led1_green'}

pmt_set_dl_delay = 0.1
pmt_set_hv_delay = 0.2
pmt_set_hv_enable_delay = 1.0   # ~600ms settling time were measured


async def xiu20_adjustment(channel='pmt1', identification=''):

    timestamp = datetime.now()
    
    channel = str(channel).lower() if (channel != '') else 'pmt1'
    identification = str(identification).upper() if (identification != '') else timestamp.strftime('%Y%m%d_%H%M%S')

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','pmt3'))

    if not channel:
        raise ValueError(f"channel must at least contain 'pmt1', 'pmt2' or 'pmt3'.")
    if not re.match('[A-Z0-9_-]+', identification):
        raise ValueError(f"identification must only contain [A-Z0-9_-] characters.")

    for ch in channel:
        if (f"{ch}_pdd_scaling" not in config) or (f"{ch}_led_scaling" not in config):
            raise Exception(f"Test bench calibration is missing")

    report_file = os.path.join(report_path, 'pmt_adjustment.csv')
    
    GlobalVar.set_stop_gc(False)
    await send_to_gc(f"Starting PMT adjustment")

    with open(report_file, 'w') as report:
        report.write(f"ID:   {identification}\n")
        report.write(f"Date: {timestamp.strftime('%Y-%m-%d %H:%M:%S')}\n")
        report.write('\n')

    await send_to_gc(f"Starting Firmware")
    await asyncio.gather(
        fmb_unit.StartFirmware(),
        eef_unit.StartFirmware(),
    )
    if GlobalVar.get_stop_gc():
        return f"pmt_adjustment stopped by user"

    ### PMT Stabilization ######################################################

    temperature = 18    # stabilization temperature in °C
    duration    = 5    # stabilization duration in minutes

    await send_to_gc(f"Stabilizing PMT")

    if 'pmt1' in channel:
        await pmt1_cooling.InitializeDevice()
        await pmt1_cooling.set_target_temperature(temperature)
        await pmt1_cooling.enable()
    if 'pmt2' in channel:
        await pmt2_cooling.InitializeDevice()
        await pmt2_cooling.set_target_temperature(temperature)
        await pmt2_cooling.enable()
    if 'pmt3' in channel:
        await uslum_fan.InitializeDevice()
        await uslum_fan.enable()
        await eef_endpoint.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE, 0)
        await meas_endpoint.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 0)
        
    for minute in range(duration):
        await send_to_gc(f"{duration - minute} min remaining...")
        for second in range(60):
            await asyncio.sleep(1)
            if GlobalVar.get_stop_gc():
                return f"pmt_adjustment stopped by user"
            
    await send_to_gc(f" ")
        
    ### Discriminator Adjustment ###############################################

    hts_laser_power = 1.0  # 1.0 is maximum and current default setting
    hts_tec_power   = 0.5  # 0.0 is maximum!!!

    await send_to_gc(f"Adjusting Discriminator Level:")

    if 'pmt3' in channel:
        await eef_endpoint.SetAnalogOutput(EEFAnalogOutput.HTSALPHATEC, hts_tec_power)
        await meas_endpoint.SetParameter(MeasurementParameter.HTSAlphaLaserPower, hts_laser_power)
        await eef_endpoint.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE, 1)
        await meas_endpoint.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 1)

    results = await xiu20_adjust_discriminator(channel, dl_start=0.0, dl_stop=1.0, dl_step=0.001, report_file=report_file)

    if 'pmt3' in channel:
        await eef_endpoint.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE, 0)
        await meas_endpoint.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 0)

    if GlobalVar.get_stop_gc():
        return f"pmt_adjustment stopped by user"

    dl = results

    await send_to_gc(f" ")
    await send_to_gc(f"dl = {dl}")
    await send_to_gc(f" ")

    ### High Voltage Adjustment ################################################

    await send_to_gc(f"Adjusting High Voltage Setting:")

    results = await xiu20_adjust_high_voltage(channel, dl, hv_start=0.3, hv_stop=0.7, hv_step=0.005, report_file=report_file)

    if GlobalVar.get_stop_gc():
        return f"pmt_adjustment stopped by user"

    hv = results['hv']          # Adjusted High Voltage Setting
    ppr_ns = results['ppr_ns']  # Adjusted Pulse Pair Resolution

    await send_to_gc(f" ")
    await send_to_gc(f"hv = {hv}, ppr_ns = {ppr_ns}")
    await send_to_gc(f" ")

    for ch in channel:
        if not (1.0 <= ppr_ns[ch] <= 24.0):
            raise Exception(f"Failed to adjust Pulse Pair Resolution of {ch.upper()} (Limits: 1.0 <= ppr_ns <= 24.0)")

    ### Analog Adjustment ######################################################

    if ('pmt1' in channel) or ('pmt2' in channel):
        
        await send_to_gc(f"Adjusting Analog Scaling:")

        results = await xiu20_adjust_analog(channel, dl, hv, ppr_ns, report_file=report_file)

        if GlobalVar.get_stop_gc():
            return f"pmt_adjustment stopped by user"

        als = results['als']    # Adjusted Analog Low Scaling
        ahs = results['ahs']    # Adjusted Analog High Scaling

        await send_to_gc(f" ")
        await send_to_gc(f"als = {als}, ahs = {ahs}")
        await send_to_gc(f" ")

        for ch in channel:
            if not (0.05 <= als[ch] <= 0.5):
                raise Exception(f"Failed to adjust Analog Low Scaling of {ch.upper()} (Limits: 0.05 <= als <= 0.5)")
            if not (90.0 <= ahs[ch] <= 500.0):
                raise Exception(f"Failed to adjust Analog High Scaling of {ch.upper()} (Limits: 90.0 <= ahs <= 500.0)")
    
    pmt_name = {'pmt1':'PMT1', 'pmt2':'PMT2', 'pmt3':'PMT_USLUM'}

    await send_to_gc(f"Adjusted Parameter:")
    for ch in channel:
        await send_to_gc(f" ")
        await send_to_gc(f"{pmt_name[ch]}")
        if ch in ['pmt1', 'pmt2']:
            await send_to_gc(f"|-----|----------|")
            await send_to_gc(f"| DL  | {dl[ch]:8.3f} | ")
            await send_to_gc(f"| HV  | {hv[ch]:8.3f} |")
            await send_to_gc(f"| PPR | {ppr_ns[ch]:5.2f}e-9 |")
            await send_to_gc(f"| ACE | {als[ch]:8.6f} |")
            await send_to_gc(f"| AHS | {ahs[ch]:8.4f} |")
            await send_to_gc(f"|-----|----------|")
        if ch in ['pmt3']:
            await send_to_gc(f"|------------|------------|--------------------|")
            await send_to_gc(f"| HV setting | DL values  | PMT puls pair res. |")
            await send_to_gc(f"|------------|------------|--------------------|")
            await send_to_gc(f"| {hv[ch]:10.3f} | {dl[ch]:10.3f} | {ppr_ns[ch]:15.2f}e-9 |")
            await send_to_gc(f"|------------|------------|--------------------|")

    with open(report_file, 'a') as report:
        report.write(f"Adjusted Parameter:\n")
        for ch in channel:
            report.write('\n')
            report.write(f"DiscriminatorLevel_{pmt_name[ch]} = {dl[ch]:.3f}\n")
            report.write(f"HighVoltageSetting_{pmt_name[ch]} = {hv[ch]:.3f}\n")
            report.write(f"Pulse_pair_res_{pmt_name[ch]}_s = {ppr_ns[ch]:.2f}e-9\n")
            if ch in ('pmt1', 'pmt2'):
                report.write(f"AnalogCountingEquivalent_{pmt_name[ch]} = {als[ch]:.6f}\n")
                report.write(f"AnalogHighRangeScale_{pmt_name[ch]} = {ahs[ch]:.3f}\n")

    shutil.copy(report_file, os.path.join(archive_path, f"{identification}_Adjustment.csv"))


async def xiu20_adjust_discriminator(channel, dl_start, dl_stop, dl_step, report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','pmt3'))

    window_ms = 100.0
    window_count = 1
    iterations = 1

    dl_offset = {'pmt1':0.05, 'pmt2':0.05, 'pmt3':0.075}        # DL is set at a fixed offset to the noise peak of the scan.
    dl_width_limit = {'pmt1':0.04, 'pmt2':0.04, 'pmt3':0.06}    # The width of the noise peak must not be larger than this limit.

    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pmt_adjust_discriminator.csv")

    with open(report_file, 'a') as report:
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        report.write(f"pmt_adjust_discriminator(channel={channel}, dl_start={dl_start:.3f}, dl_stop={dl_stop:.3f}, dl_step={dl_step:.3f}) started at {timestamp}\n")
        report.write(f"temperature: {await pmt_get_temperature(channel)}\n")
        report.write('\n')
    
        cps = {}    # Counts Per Second (The measured noise)
        dt = {}     # Dead Time (The amount of time the PMT signal was above the discriminator level)
        
        dl_range = np.arange(dl_start, (dl_stop + 1e-6), dl_step).round(6)  # The discriminator level scan range

        output = f"dl    ; "
        for ch in channel:
            cps[ch] = np.zeros_like(dl_range)
            dt[ch] = np.zeros_like(dl_range)

            output += f"{ch}_cps   ; {ch}_dt    ; "

        await send_to_gc(output, report=report)

        await pmt_set_hv_enable(channel, 0)

        for i, dl in enumerate(dl_range):
            await pmt_set_dl(channel, dl)
            await asyncio.sleep(pmt_set_dl_delay)
            results = await pmt_counting_measurement(window_ms, window_count, iterations)
            if GlobalVar.get_stop_gc():
                return f"pmt_adjust_discriminator stopped by user"

            output = f"{dl:5.3f} ; "
            for ch in channel:
                cps[ch][i] = results[f"{ch}_cps_mean"]
                dt[ch][i] = results[f"{ch}_dt_mean"]

                output += f"{cps[ch][i]:10.0f} ; {dt[ch][i]:10.0f} ; "

            await send_to_gc(output, report=report)

        report.write('\n')

        plot_min = 1.0
        plot_max = 0.0

        dl = {}

        for ch in channel:
            cps_max = np.max(cps[ch])
            dt_max = np.max(dt[ch])

            dl_peak = dl_range[cps[ch] > (cps_max * 0.9)]
            dt_peak = dl_range[dt[ch] > (dt_max * 0.5)]

            if (len(dl_peak) > 0) and (cps_max >= 100):
                dl_center = (np.max(dl_peak) + np.min(dl_peak)) / 2
            elif (len(dt_peak > 0)):
                dl_center = np.max(dt_peak)
            else:
                raise Exception(f"Failed to adjust discriminator level. No peak found in the measurement.")

            dl[ch] = np.round((dl_center + dl_offset[ch]), 3)
        
            report.write(f"{ch}_dl ; {dl[ch]:.3f} ; ")

            plot_min = min(plot_min, (dl_center - dl_offset[ch]))
            plot_max = max(plot_max, (dl_center + 2 * dl_offset[ch]))

            #TODO Filter outlier counts when zero before and after.
            dl_noise = dl_range[cps[ch] > 0.0]
            dl_width = np.max(dl_noise) - dl_center if (len(dl_noise) > 0) else 0.0

            # if dl_width > dl_width_limit[ch]:
            #     raise Exception(f"Failed to adjust discriminator level. Peak is too wide. (center: {dl_center:.3f}, width: {dl_width:.3f})")

        report.write('\n\n')

        # Plot only the relevant part of the scan.
        plot_select = np.logical_and((dl_range >= plot_min), (dl_range <= plot_max))
        dl_range = dl_range[plot_select]
        for ch in channel:
            cps[ch] = cps[ch][plot_select]
            dt[ch] = dt[ch][plot_select]
        
        await plot_dl_scan(channel, dl_range, cps, dt, dl, file_name='dl_scan.png')

    return dl


async def xiu20_adjust_high_voltage(channel, dl, hv_start, hv_stop, hv_step, scan_type='default', report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','pmt3'))

    scan_type = str(scan_type) if (scan_type != '') else 'default'

    linearity_limit = 0.9997    # Select the highest high voltage setting with an r² value greater or equal to this limit.
    linearity_stop  = 0.997     # Stop the scan when linearity drops below this limit.

    rel_error_limit = 0.10      # Select the highest high voltage setting with an relative error less or equal to this limit.
    rel_error_stop  = 0.25      # Stop the scan when relative error rises above this limit.

    cps_min = 1000

    if scan_type == 'default':
        window_ms = 50
        window_count = 20
        selection_criterion = 'rel_error_limit'     # Select highest HV within the relative error limit.
        # selection_criterion = 'linearity_limit'   # Select highest HV within the linearity limit.
    if scan_type == 'cal_hv_scan':
        window_ms = 50
        window_count = 20
        selection_criterion = 'linearity_max'       # Select HV with the highest linearity.
        # selection_criterion = 'rel_error_min'     # Select HV with the lowest relative error.

    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pmt_adjust_high_voltage.csv")

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    
    ppr_ns = {}         # The pulse pair resolution in ns.
    sensitivity = {}    # The measured sensitivity.
    gain = {}           # The signal gain of the PMT.
    linearity = {}      # The measured linearity (r² value).
    rel_error = {}      # The measured relative error.

    hv_start = await pmt_get_hv_start(channel, dl, hv_start, hv_stop, hv_step, cps_min)
    if GlobalVar.get_stop_gc():
        return f"pmt_adjust_high_voltage stopped by user"
    
    hv_range = np.arange(hv_start, (hv_stop + 1e-6), hv_step).round(6)  # The high voltage setting scan range.

    output = f"hv    ; "
    for ch in channel:
        ppr_ns[ch] = np.zeros_like(hv_range)
        sensitivity[ch] = np.zeros_like(hv_range)
        gain[ch] = np.zeros_like(hv_range)
        linearity[ch] = np.zeros_like(hv_range)
        rel_error[ch] = np.ones_like(hv_range)      # Assuming rel_error_stop < 1.0

        output += f"{ch}_ppr ; {ch}_sen ; {ch}_gain ; {ch}_lin ; {ch}_err ; "

    await send_to_gc(output)

    for i, hv in enumerate(hv_range):
        results = await xiu20_signal_scan(channel, dl, hv, window_ms, window_count, scan_type, report_file=report_file)
        if GlobalVar.get_stop_gc():
            return f"pmt_adjust_high_voltage stopped by user"

        continue_scan = False

        output = f"{hv:5.3f} ; "
        for ch in channel:
            ppr_ns[ch][i] = results['ppr_ns'][ch]
            sensitivity[ch][i] = results['sensitivity'][ch]
            gain[ch][i] = results['gain'][ch]
            linearity[ch][i] = results['linearity'][ch]
            rel_error[ch][i] = results['rel_error'][ch]

            output += f"{ppr_ns[ch][i]:5.2f}e-9 ; {sensitivity[ch][i]:8.3f} ; {gain[ch][i]:9.3f} ; {linearity[ch][i]:8.6f} ; {rel_error[ch][i]:8.2%} ; "

            if ('linearity' in selection_criterion) and ((linearity[ch][i] >= linearity_stop) or (np.max(linearity[ch]) < linearity_limit)):
                continue_scan = True
            if ('rel_error' in selection_criterion) and ((rel_error[ch][i] <= rel_error_stop) or (np.min(rel_error[ch]) > rel_error_limit)):
                continue_scan = True

        await send_to_gc(output)

        if not continue_scan:
            hv_range = hv_range[:(i + 1)]
            for ch in channel:
                ppr_ns[ch] = ppr_ns[ch][:(i + 1)]
                sensitivity[ch] = sensitivity[ch][:(i + 1)]
                linearity[ch] = linearity[ch][:(i + 1)]
                rel_error[ch] = rel_error[ch][:(i + 1)]
            break

    with open(report_file, 'a') as report:
        report.write(f"pmt_adjust_high_voltage(channel={channel}, dl={dl}, hv_start={hv_start:.3f}, hv_stop={hv_stop:.3f}, hv_step={hv_step:.3f}, scan_type={scan_type}) started at {timestamp}\n")
        report.write('\n')

        report.write(f"hv    ; ")
        for ch in channel:
            report.write(f"{ch}_ppr ; {ch}_sen ; {ch}_gain ; {ch}_lin ; {ch}_err ; ")
        report.write('\n')

        for i, hv in enumerate(hv_range):
            report.write(f"{hv:5.3f} ; ")
            for ch in channel:
                report.write(f"{ppr_ns[ch][i]:5.2f}e-9 ; {sensitivity[ch][i]:8.3f} ; {gain[ch][i]:9.3f} ; {linearity[ch][i]:8.6f} ; {rel_error[ch][i]:8.2%} ; ")
            report.write('\n')

        report.write('\n')

        hv = {}
        for ch in channel:
            if 'linearity' in selection_criterion:
                hv_range_linear = hv_range[linearity[ch] >= linearity_limit]
            if 'rel_error' in selection_criterion:
                hv_range_linear = hv_range[rel_error[ch] <= rel_error_limit]

            if hv_range_linear.size < 1:
                # raise Exception(f"Failed to adjust high voltage setting of {ch.upper()}. Required linearity not reached.")
                hv[ch] = 0.475 #0.525
                await send_to_gc(f" ")
                await send_to_gc(f"Failed to adjust high voltage setting of {ch.upper()}. Required linearity not reached. Using default value {hv[ch]:.3f}.")
            else:
                if selection_criterion == 'linearity_limit' or selection_criterion == 'rel_error_limit':
                    hv[ch] = np.max(hv_range_linear)
                if selection_criterion == 'linearity_max':
                    hv[ch] = hv_range[linearity[ch] == np.max(linearity[ch])][0]
                if selection_criterion == 'rel_error_min':
                    hv[ch] = hv_range[rel_error[ch] == np.min(rel_error[ch])][0]

            ppr_ns[ch] = ppr_ns[ch][hv_range == hv[ch]][0]

            report.write(f"{ch}_hv ; {hv[ch]:.3f} ; {ch}_ppr_ns ; {ppr_ns[ch]:.2f} ; ")
        
        report.write('\n')
        report.write('\n')

    return {'hv':hv, 'ppr_ns':ppr_ns}


async def pmt_get_hv_start(channel, dl, hv_start=0.3, hv_stop=0.7, hv_step=0.005, cps_min=1000):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','pmt3'))

    for led in np.unique(list(led_dimmed.values()) + list(led_bright.values())):
        led_source, led_channel, led_type = led.split('_')
        await set_led_current(0, led_source, led_channel, led_type)

    await pmt_set_dl(channel, dl)
    await pmt_set_hv(channel, hv_start)
    await asyncio.sleep(pmt_set_hv_delay)
    await pmt_set_hv_enable(channel, 1)
    await asyncio.sleep(pmt_set_hv_enable_delay)

    try:
        for ch in channel:
            led_source, led_channel, led_type = led_dimmed[ch].split('_')
            await set_led_current(1, led_source, led_channel, led_type)

        hv_range = np.arange(hv_start, (hv_stop + 1e-6), hv_step).round(6)

        for hv in hv_range:
            await pmt_set_hv(channel, hv)
            await asyncio.sleep(0.5)
            results = await pmt_counting_measurement(window_ms=100, window_count=1, iterations=1)
            if GlobalVar.get_stop_gc():
                return f"pmt_get_hv_start stopped by user"
            
            for ch in channel:
                if results[f"{ch}_cps_mean"] >= cps_min:
                    return round(hv, 6)

            await send_to_gc(f"hv={hv:.3f} skipped (pmt1={results['pmt1_cps_mean']}, pmt2={results['pmt2_cps_mean']}, pmt3={results['pmt3_cps_mean']})", log=True)

        raise Exception(f"The count rate is below the limit for all high voltage settings.")

    finally:
        await pmt_set_hv_enable(channel, 0)


async def xiu20_signal_scan(channel, dl, hv, window_ms=50, window_count=20, scan_type='default', report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','pmt3'))
    
    scan_type = str(scan_type) if (scan_type != '') else 'default'

    if scan_type == 'default':
        dark_iterations = 0 #20
        led_current_dimmed = [1,2,3,4,5,7,10,13,18,24,33,45,63,88,124]
        #TODO Unify signal range for new test benches
        if 'pmt3' in channel:
            led_current_bright = [2,3,4,5,7,10,13,18,25,33,46,64,90,127]
        else:
            led_current_bright = [2,3,4,5,7,10,13,18,25,33,46,64]
        lin_reg_size = len(led_current_dimmed)
        sen_ref_index = len(led_current_dimmed)
        linearization = 'r_squared_min'
        # linearization = 'rel_error_max'
    if scan_type == 'cal_hv_scan':
        dark_iterations = 0
        led_current_dimmed = []
        led_current_bright = [1,2,3,4,5,7,10,13,18,25,33,46,64,90,127]
        lin_reg_size = 2
        sen_ref_index = 0
        linearization = 'r_squared_min'
        # linearization = 'rel_error_max'

    scan_size = len(led_current_dimmed) + len(led_current_bright)

    dark_cps = {}       # The mean value of the dark measurement in counts per second.
    dark_std = {}       # The standard deviation of the dark measurement in counts per second.
    signal_cps = {}     # The measured signal values of the pmt in counts per second.
    signal_ref = {}     # The measured signal values of the photodiode.
    
    signal_al = {}
    signal_ah = {}
    
    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pmt_signal_scan.csv")

    with open(report_file, 'a') as report:
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        report.write(f"pmt_signal_scan(channel={channel}, dl={dl}, hv={hv}, window_ms={window_ms}, window_count={window_count}, scan_type={scan_type}) started at {timestamp}\n")
        report.write(f"temperature: {await pmt_get_temperature(channel)}\n")
        report.write('\n')

        for led in np.unique(list(led_dimmed.values()) + list(led_bright.values())):
            led_source, led_channel, led_type = led.split('_')
            await set_led_current(0, led_source, led_channel, led_type)

        await pmt_set_dl(channel, dl)
        await pmt_set_hv(channel, hv)
        await asyncio.sleep(pmt_set_hv_delay)
        await pmt_set_hv_enable(channel, 1)
        await asyncio.sleep(pmt_set_hv_enable_delay)

        try:
            if dark_iterations > 0:
                results = await pmt_counting_measurement(window_ms, window_count, dark_iterations)
                if GlobalVar.get_stop_gc():
                    await pmt_set_hv_enable(channel, 0)
                    return f"pmt_signal_scan stopped by user"

                for ch in channel:
                    dark_cps[ch] = results[f"{ch}_cps_mean"]
                    dark_std[ch] = results[f"{ch}_cps_std"]

                    report.write(f"{ch}_dark_cps ; {dark_cps[ch]:.6f} ; {ch}_dark_std ; {dark_std[ch]:.6f} ; ")

                report.write('\n')
                report.write('\n')

            for ch in channel:
                signal_cps[ch] = np.zeros(scan_size)
                signal_ref[ch] = np.zeros(scan_size)

                signal_al[ch] = np.zeros(scan_size)
                signal_ah[ch] = np.zeros(scan_size)

                report.write(f"{ch}_signal ; {ch}_cps    ; {ch}_al     ; {ch}_ah     ; ")

            report.write('\n')

            ppr_ns = {ch:0.0 for ch in channel}
            sensitivity = {ch:0.0 for ch in channel}
            gain = {ch:0.0 for ch in channel}
            linearity = {ch:0.0 for ch in channel}
            rel_error = {ch:1.0 for ch in channel}

            index = 0

            for current in led_current_dimmed:
                for ch in channel:
                    led_source, led_channel, led_type = led_dimmed[ch].split('_')
                    await set_led_current(current, led_source, led_channel, led_type)

                results = await pmt_counting_measurement(window_ms, window_count, iterations=1)
                if GlobalVar.get_stop_gc():
                    return f"pmt_signal_scan stopped by user"

                for ch in channel:
                    signal_cps[ch][index] = results[f"{ch}_cps_mean"]
                    signal_ref[ch][index] = results[f"{ch}_pdd_mean"] * config[f"{ch}_led_scaling"]

                    signal_al[ch][index] = results[f"{ch}_al_mean"]
                    signal_ah[ch][index] = results[f"{ch}_ah_mean"]

                    report.write(f"{signal_ref[ch][index]:11.0f} ; {signal_cps[ch][index]:11.0f} ; {signal_al[ch][index]:11.0f} ;  {signal_ah[ch][index]:11.0f} ; ")

                report.write('\n')
                index += 1
            
            for ch in channel:
                led_source, led_channel, led_type = led_dimmed[ch].split('_')
                await set_led_current(0, led_source, led_channel, led_type)

            for current in led_current_bright:
                for ch in channel:
                    led_source, led_channel, led_type = led_bright[ch].split('_')
                    await set_led_current(current, led_source, led_channel, led_type)

                results = await pmt_counting_measurement(window_ms, window_count, iterations=1)
                if GlobalVar.get_stop_gc():
                    return f"pmt_signal_scan stopped by user"

                for ch in channel:
                    signal_cps[ch][index] = results[f"{ch}_cps_mean"]
                    signal_ref[ch][index] = results[f"{ch}_pdd_mean"]

                    signal_al[ch][index] = results[f"{ch}_al_mean"]
                    signal_ah[ch][index] = results[f"{ch}_ah_mean"]

                    report.write(f"{signal_ref[ch][index]:11.0f} ; {signal_cps[ch][index]:11.0f} ; {signal_al[ch][index]:11.0f} ;  {signal_ah[ch][index]:11.0f} ; ")

                report.write('\n')
                index += 1
            
            for ch in channel:
                led_source, led_channel, led_type = led_bright[ch].split('_')
                await set_led_current(0, led_source, led_channel, led_type)

        finally:
            await pmt_set_hv_enable(channel, 0)

        report.write('\n')

        for ch in channel:
            ppr_ns[ch], linearity[ch], rel_error[ch] = pmt_calculate_ppr_ns(signal_ref[ch], signal_cps[ch], optimization=linearization)
            
            signal_cps[ch] = pmt_calculate_correction(signal_cps[ch], ppr_ns[ch])

            if (dark_iterations > 0) and (dark_cps[ch] < signal_cps[ch][sen_ref_index]):
                sensitivity[ch] = 3 * signal_ref[ch][sen_ref_index] * dark_std[ch] / (signal_cps[ch][sen_ref_index] - dark_cps[ch])
            else:
                sensitivity[ch] = 0.0
            
            gain[ch] = np.mean(signal_cps[ch][:lin_reg_size] / signal_ref[ch][:lin_reg_size])

            signal_ref[ch] = np.mean(signal_cps[ch][:lin_reg_size] / signal_ref[ch][:lin_reg_size]) * signal_ref[ch]

            report.write(f"{ch}_ppr ; {ppr_ns[ch]:.2f}e-9 ; {ch}_sen ; {sensitivity[ch]:.3f} ; {ch}_gain ; {gain[ch]:.3f} ; {ch}_lin ; {linearity[ch]:.6f} ; {ch}_err ; {rel_error[ch]:.2%} ; ")
            
        report.write('\n')
        report.write('\n')

        await plot_signal_scan(channel, signal_ref, signal_cps, signal_ref, hv, file_name=f"signal_scan_hv_{hv:.3f}.png")

    return {'ppr_ns':ppr_ns, 'sensitivity':sensitivity, 'gain':gain, 'linearity':linearity, 'rel_error':rel_error}


async def xiu20_adjust_analog(channel, dl, hv, ppr_ns, report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2'))
    
    iterations = 100

    led = {'pmt1':'fmb_led2_green', 'pmt2':'fmb_led4_green'}
    led_current = np.linspace(1, 127, 127)
    
    al_lower_limit = (65536 - 1310) * 0.2     # The analog low value must be greater or equal to this limit. 20% of the analog low range with ~100 mV offset voltage.
    al_upper_limit = (65536 - 1310) * 0.8     # The analog low value must be less or equal to this limit. 80% of the analog low range with ~100 mV offset voltage.
    
    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pmt_adjust_analog.csv")

    with open(report_file, 'a') as report:
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        report.write(f"pmt_adjust_analog(channel={channel}, dl={dl}, hv={hv}, ppr_ns={ppr_ns}) started at {timestamp}\n")
        report.write(f"temperature: {await pmt_get_temperature(channel)}\n")
        report.write('\n')

        await pmt_set_dl(channel, dl)
        await pmt_set_hv(channel, hv)
        await asyncio.sleep(pmt_set_hv_delay)
        await pmt_set_hv_enable(channel, 1)
        await asyncio.sleep(pmt_set_hv_enable_delay)
        
        # Determine the required measurement window time in ms.
        window_ms = 1.0
        for ch in channel:
            led_source, led_channel, led_type = led[ch].split('_')
            await set_led_current(np.max(led_current), led_source, led_channel, led_type)

        results = await pmt_multi_range_measurement(window_ms, iterations)

        al_max = 0.0
        for ch in channel:
            al_max = max(al_max, results[f"{ch}_al_mean"])
        if (al_max > al_upper_limit):
            window_ms = 0.1
            results = await pmt_multi_range_measurement(window_ms, iterations)
        
        al_max = 0.0
        for ch in channel:
            al_max = max(al_max, results[f"{ch}_al_mean"])
        if (al_max > al_upper_limit):
            raise Exception(f"Failed to adjust analog scaling factors. Signal is too bright. (max analog low: {al_max:.0f})")
        if (al_max < (al_upper_limit / 10000 * iterations)):
            raise Exception(f"Failed to adjust analog scaling factors. Signal is too dark. (max analog low: {al_max:.0f})")

        window_ms = np.round(al_upper_limit / al_max * window_ms, 3)

        report.write(f"window_ms ;  {window_ms:.3f}\n")
        report.write('\n')
        
        cnt = {}    # The measured couting values.
        al = {}     # The measured analog low values.
        ah = {}     # The measured analog high values.
        
        output = f"signal ; "
        for ch in channel:
            cnt[ch] = np.zeros_like(led_current)
            al[ch] = np.zeros_like(led_current)
            ah[ch] = np.zeros_like(led_current)

            output += f"{ch}_cnt ; {ch}_al  ; {ch}_ah  ; "

        await send_to_gc(output, report=report)

        for i, current in enumerate(led_current):
            for ch in channel:
                led_source, led_channel, led_type = led[ch].split('_')
                await set_led_current(current, led_source, led_channel, led_type)

            results = await pmt_multi_range_measurement(window_ms, iterations)
            if GlobalVar.get_stop_gc():
                return f"pmt_adjust_analog stopped by user"

            output = f"{led_current[i]:6.0f} ; "
            for ch in channel:
                cnt[ch][i] = results[f"{ch}_cnt_mean"]
                al[ch][i] = results[f"{ch}_al_mean"]
                ah[ch][i] = results[f"{ch}_ah_mean"]

                output += f"{cnt[ch][i]:8.0f} ; {al[ch][i]:8.0f} ; {ah[ch][i]:8.0f} ; "
            
            await send_to_gc(output, report=report)
            
        report.write('\n')

        for ch in channel:
            led_source, led_channel, led_type = led[ch].split('_')
            await set_led_current(0, led_source, led_channel, led_type)
        
        await pmt_set_hv_enable(channel, 0)
    
        if isinstance(ppr_ns, float):
            ppr_ns = {'pmt1':ppr_ns, 'pmt2':ppr_ns, 'pmt3':ppr_ns}
        
        als = {}    # The adjusted analog low scaling.
        ahs = {}    # The adjsuted analog high scaling.

        for ch in channel:
            # Count Rate Correction
            # ppr_ms = ppr_ns[ch] * 1e-6
            # cnt[ch] = cnt[ch] / (1 - cnt[ch] * ppr_ms / window_ms)

            cnt[ch] = pmt_calculate_correction(cnt[ch], ppr_ns[ch], window_ms)

            limits = np.logical_and((al[ch] >= al_lower_limit), (al[ch] <= al_upper_limit))
            als[ch] = np.sum(cnt[ch][limits] * al[ch][limits]) / np.sum(al[ch][limits] ** 2)
            ahs[ch] = np.sum(al[ch][limits] * ah[ch][limits]) / np.sum(ah[ch][limits] ** 2)

            al[ch] = al[ch] * als[ch]
            ah[ch] = ah[ch] * ahs[ch] * als[ch]
            
            report.write(f"{ch}_als ; {als[ch]} ; {ch}_ahs ; {ahs[ch]} ; ")
            
        report.write('\n')
        report.write('\n')

        await plot_analog_scan(channel, led_current, cnt, al, ah, file_name='analog_scan.png')

    return {'als':als, 'ahs':ahs}


async def plot_dl_scan(channel, dl_range, cps, dt, dl, file_name='graph.png'):

    plt.clf()

    for i, ch in enumerate(channel):
        plt.subplot(len(channel), 2, (i * 2 + 1))
        if i == 0:
            plt.title('Counts Per Second')
        if i == (len(channel) - 1):
            plt.xlabel('discriminator level')
        plt.ylabel(ch.upper())
        plt.yscale('symlog', linthresh=1)
        plt.plot(dl_range, cps[ch])
        plt.axvline(dl[ch], color='r')

        plt.subplot(len(channel), 2, (i * 2 + 2))
        if i == 0:
            plt.title('Dead Time')
        if i == (len(channel) - 1):
            plt.xlabel('discriminator level')
        plt.plot(dl_range, dt[ch])
        plt.axvline(dl[ch], color='r')

    plt.savefig(os.path.join(images_path, file_name))
    await send_gc_event('RefreshGraph', file_name=os.path.join('pmt_adjust_images', file_name))


async def plot_signal_scan(channel, signal, cps, cps_ref, hv, file_name='graph.png'):
    
    if isinstance(hv, float):
        hv = {'pmt1':hv, 'pmt2':hv, 'pmt3':hv}

    plt.clf()

    for i, ch in enumerate(channel):
        plt.subplot(len(channel), 1, (i + 1))
        if i == 0:
            plt.title(f"Signal Scan HV={hv[ch]:.3f}")
        if i == (len(channel) - 1):
            plt.xlabel('signal')
        plt.ylabel(ch.upper())
        plt.xscale('log')
        plt.yscale('log')
        plt.plot(signal[ch], cps_ref[ch], label='reference', color='r')
        plt.plot(signal[ch], cps[ch], label='measured', color='b')
        plt.legend(loc='upper left')

    plt.savefig(os.path.join(images_path, file_name))
    await send_gc_event('RefreshGraph', file_name=os.path.join('pmt_adjust_images', file_name))


async def plot_analog_scan(channel, current, cnt, al, ah, file_name='graph.png'):

    plt.clf()

    for i, ch in enumerate(channel):
        plt.subplot(len(channel), 1, (i + 1))
        if i == 0:
            plt.title(f"Analog Adjustment")
        plt.ylabel(ch.upper())
        if ah[ch] is not None:
            plt.plot(current, ah[ch], label='analog_high', color='g')
        if al[ch] is not None:
            plt.plot(current, al[ch], label='analog_low', color='r')
        if cnt[ch] is not None:
            plt.plot(current, cnt[ch], label='counting', color='b')
        plt.legend(loc='upper left')

    plt.savefig(os.path.join(images_path, file_name))
    await send_gc_event('RefreshGraph', file_name=os.path.join('pmt_adjust_images', file_name))


async def plot_pdd_scan(channel, pmt_signal, pdd_measured, pdd_scaled, file_name='graph.png'):

    plt.clf()

    for i, ch in enumerate(channel):
        plt.subplot(len(channel), 1, (i + 1))
        if i == 0:
            plt.title(f"Photodiode Signal Scan")
        if i == (len(channel) - 1):
            plt.xlabel('pmt_signal')
        plt.ylabel(f"PDD of {ch.upper()}")
        plt.xscale('log')
        plt.yscale('log')
        plt.plot(pmt_signal[ch], pdd_measured[ch], label='pdd_measured', color='r')
        plt.plot(pmt_signal[ch], pdd_scaled[ch], label='pdd_scaled', color='b')
        plt.legend(loc='upper left')

    plt.savefig(os.path.join(images_path, file_name))
    await send_gc_event('RefreshGraph', file_name=os.path.join('pmt_adjust_images', file_name))


def pmt_calculate_ppr_ns(signal, counts, window_ms=1000, optimization='r_squared_min', optimization_arg=None):
    ppr_ns_max = 25.0
    ppr_ns = 0.0

    linearity, rel_error = pmt_calculate_linearity(signal, counts, ppr_ns, window_ms)

    precision = 2
    for step_ns in [1.0 / 10**digit for digit in range(precision + 1)]:

        if ppr_ns >= (step_ns * 10.0):
            # Take one step back for cases like this: Optimal ppr is 11.5 and r²(11.0) < r²(12.0) > r²(13.0).
            ppr_ns = round((ppr_ns - step_ns * 10.0), precision)
            linearity, rel_error = pmt_calculate_linearity(signal, counts, ppr_ns, window_ms)

        stop = False
        while not stop:
            temp_ppr_ns = round((ppr_ns + step_ns), precision)
            temp_linearity, temp_rel_error = pmt_calculate_linearity(signal, counts, temp_ppr_ns, window_ms)

            if optimization == 'r_squared':
                index = int(optimization_arg) if (optimization_arg is not None) else (len(signal) - 1)
                if (index < 2) or (index >= len(signal)):
                    raise ValueError(f"Invalid optimization signal index.")
                stop = temp_linearity[index] < linearity[index]
            elif optimization == 'r_squared_min':
                stop = np.min(temp_linearity) < np.min(linearity)
            elif optimization == 'rel_error_max':
                stop = np.max(temp_rel_error) > np.max(rel_error)
            else:
                raise ValueError(f"Invalid optimization option.")

            if not stop:
                ppr_ns = temp_ppr_ns
                linearity = temp_linearity
                rel_error = temp_rel_error

            if ppr_ns > ppr_ns_max:
                # Failed to calculate pulse pair resolution
                return (0.0, 0.0, 0.0)

    return (ppr_ns, np.min(linearity), np.max(rel_error))


def pmt_calculate_linearity(signal, counts, ppr_ns, window_ms=1000):

    counts = pmt_calculate_correction(counts, ppr_ns, window_ms)

    linearity = pmt_calculate_r_squared(signal, counts)
    rel_error = pmt_calculate_rel_error(signal, counts)

    return (linearity, rel_error)


def pmt_calculate_r_squared(signal, counts):

    r_squared = np.ones_like(signal)

    for i in range(2, len(signal)):
        correlation_coef = np.corrcoef(signal[:(i + 1)], counts[:(i + 1)])
        r_value = correlation_coef[0,1]
        r_squared[i] = r_value**2

    return r_squared


def pmt_calculate_rel_error(signal, counts):

    ref_size = round(len(signal) / 2)
    
    counts_ref = np.mean(counts[:ref_size] / signal[:ref_size]) * signal
    rel_error = np.abs((counts - counts_ref) / counts_ref)

    return rel_error


def pmt_calculate_offset(signal, counts):

    trendline = np.polyfit(signal, counts, deg=1)
    counts_ref = np.polyval(trendline, signal)

    offset = np.abs(counts - counts_ref) / signal

    return offset


def pmt_calculate_correction(counts, ppr_ns, window_ms=1000):

    counts = np.array(counts)
    counts = counts / (1 - counts * ppr_ns * 1e-6 / window_ms)

    return counts


def parse_channel_parameter(channel, mask=('pmt1','pmt2','pmt3')):
    return [ch for ch in mask if ch in channel]


async def pmt_get_temperature(channel):
    temperature = ''
    if 'pmt1' in channel:
        temperature += f"pmt1={await pmt1_cooling.get_feedback_value():.2f}°C ; "
    if 'pmt2' in channel:
        temperature += f"pmt2={await pmt2_cooling.get_feedback_value():.2f}°C ; "
    if 'pmt3' in channel:
        temperature += f"pmt3={(await eef_endpoint.GetAnalogInput(EEFAnalogInput.HTSALPHATEMPIN))[0] * -31.81 + 41.992:.2f}°C ; "
    return temperature


async def pmt_set_dl(channel, dl):
    if isinstance(dl, float):
        dl = {'pmt1':dl, 'pmt2':dl, 'pmt3':dl}
    if 'pmt1' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, dl['pmt1'])
    if 'pmt2' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, dl['pmt2'])
    if 'pmt3' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMTUSLUMDiscriminatorLevel, dl['pmt3'])


async def pmt_set_hv(channel, hv):
    if isinstance(hv, float):
        hv = {'pmt1':hv, 'pmt2':hv, 'pmt3':hv}
    if 'pmt1' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, hv['pmt1'])
    if 'pmt2' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT2HighVoltageSetting, hv['pmt2'])
    if 'pmt3' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageSetting, hv['pmt3'])


async def pmt_set_hv_enable(channel, enable):
    if 'pmt1' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, enable)
    if 'pmt2' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT2HighVoltageEnable, enable)
    if 'pmt3' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageEnable, enable)


async def pmt_counting_measurement(window_ms=50.0, window_count=20, iterations=1):
    
    op_id = 'pmt_counting_measurement'
    meas_unit.ClearOperations()
    # await load_pmt_counting_measurement(op_id, window_ms, window_count)
    await load_pmt_multi_range_measurement(op_id, window_ms, window_count)

    pmt1_pdd_scaling = config.get('pmt1_pdd_scaling', 101.0)
    pmt2_pdd_scaling = config.get('pmt2_pdd_scaling', 101.0)
    pmt3_pdd_scaling = config.get('pmt3_pdd_scaling', 101.0)
    
    pmt1_cps = np.zeros(iterations)
    pmt1_dt = np.zeros(iterations)
    pmt1_pdd = np.zeros(iterations)
    pmt2_cps = np.zeros(iterations)
    pmt2_dt = np.zeros(iterations)
    pmt2_pdd = np.zeros(iterations)
    pmt3_cps = np.zeros(iterations)
    pmt3_dt = np.zeros(iterations)
    pmt3_pdd = np.zeros(iterations)
    
    pmt1_al = np.zeros(iterations)
    pmt1_ah = np.zeros(iterations)
    pmt2_al = np.zeros(iterations)
    pmt2_ah = np.zeros(iterations)

    for i in range(iterations):
        if GlobalVar.get_stop_gc():
            return f"pmt_counting_measurement stopped by user"
        
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        pmt1_cps[i] = (results[0]  + (results[1]  << 32)) / window_count / window_ms * 1000.0
        pmt1_dt[i]  = (results[2]  + (results[3]  << 32)) / window_count / window_ms * 1000.0
        pmt2_cps[i] = (results[6]  + (results[7]  << 32)) / window_count / window_ms * 1000.0
        pmt2_dt[i]  = (results[8]  + (results[9]  << 32)) / window_count / window_ms * 1000.0
        pmt3_cps[i] = (results[12] + (results[13] << 32)) / window_count / window_ms * 1000.0
        pmt3_dt[i]  = (results[14] + (results[15] << 32)) / window_count / window_ms * 1000.0
        pmt1_pdd[i] = (results[16] + results[17] * pmt1_pdd_scaling) / window_count / window_ms * 1000.0
        pmt2_pdd[i] = (results[18] + results[19] * pmt2_pdd_scaling) / window_count / window_ms * 1000.0
        pmt3_pdd[i] = (results[20] + results[21] * pmt3_pdd_scaling) / window_count / window_ms * 1000.0

        pmt1_al[i] = results[4] / window_count / window_ms * 1000.0
        pmt1_ah[i] = results[5] / window_count / window_ms * 1000.0
        pmt2_al[i] = results[10] / window_count / window_ms * 1000.0
        pmt2_ah[i] = results[11] / window_count / window_ms * 1000.0

    results = {}

    results['pmt1_cps_mean'] = np.mean(pmt1_cps)
    results['pmt1_dt_mean'] = np.mean(pmt1_dt)
    results['pmt1_pdd_mean'] = np.mean(pmt1_pdd)
    results['pmt2_cps_mean'] = np.mean(pmt2_cps)
    results['pmt2_dt_mean'] = np.mean(pmt2_dt)
    results['pmt2_pdd_mean'] = np.mean(pmt2_pdd)
    results['pmt3_cps_mean'] = np.mean(pmt3_cps)
    results['pmt3_dt_mean'] = np.mean(pmt3_dt)
    results['pmt3_pdd_mean'] = np.mean(pmt3_pdd)
    
    results['pmt1_al_mean'] = np.mean(pmt1_al)
    results['pmt1_ah_mean'] = np.mean(pmt1_ah)
    results['pmt2_al_mean'] = np.mean(pmt2_al)
    results['pmt2_ah_mean'] = np.mean(pmt2_ah)
    results['pmt3_al_mean'] = 0.0
    results['pmt3_ah_mean'] = 0.0
    
    results['pmt1_cps_std'] = np.std(pmt1_cps)
    results['pmt1_dt_std'] = np.std(pmt1_dt)
    results['pmt1_pdd_std'] = np.std(pmt1_pdd)
    results['pmt2_cps_std'] = np.std(pmt2_cps)
    results['pmt2_dt_std'] = np.std(pmt2_dt)
    results['pmt2_pdd_std'] = np.std(pmt2_pdd)
    results['pmt3_cps_std'] = np.std(pmt3_cps)
    results['pmt3_dt_std'] = np.std(pmt3_dt)
    results['pmt3_pdd_std'] = np.std(pmt3_pdd)
    
    results['pmt1_al_std'] = np.std(pmt1_al)
    results['pmt1_ah_std'] = np.std(pmt1_ah)
    results['pmt2_al_std'] = np.std(pmt2_al)
    results['pmt2_ah_std'] = np.std(pmt2_ah)
    results['pmt3_al_std'] = 0.0
    results['pmt3_ah_std'] = 0.0

    return results


async def load_pmt_counting_measurement(op_id, window_ms, window_count):
    if (window_ms < 0.02):
        raise ValueError(f"window_ms must be greater or equal to 0.02 ms")

    fixed_range_us = 20
    auto_range_us = round(window_ms * 1000) - fixed_range_us
    auto_range_us_coarse, auto_range_us_fine = divmod(auto_range_us, 65536)

    hv_gate_delay = 1000000     #  10 ms
    full_reset_delay = 40000    # 400 us
    us_tick_delay = 100         #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # pmt1_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # pmt1_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt1_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # pmt1_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # pmt1_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 6)  # pmt2_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 7)  # pmt2_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 8)  # pmt2_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 9)  # pmt2_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt2_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt2_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # pmt3_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # pmt3_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # pmt3_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # pmt3_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # ref_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # ref_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=18)  # abs_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=19)  # abs_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=20)  # aux_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=21)  # aux_ah

    seq_gen.TimerWaitAndRestart(hv_gate_delay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.Loop(window_count)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset, ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset, ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.low_range_reset, abs=IntegratorMode.low_range_reset, aux=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(us_tick_delay)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_autorange, abs=IntegratorMode.integrate_autorange, aux=IntegratorMode.integrate_autorange)
    if auto_range_us_coarse > 0:
        seq_gen.Loop(auto_range_us_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(us_tick_delay)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if auto_range_us_fine > 0:
        seq_gen.Loop(auto_range_us_fine)
        seq_gen.TimerWaitAndRestart(us_tick_delay)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_with_fixed_range, abs=IntegratorMode.integrate_with_fixed_range, aux=IntegratorMode.integrate_with_fixed_range)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)
    
    seq_gen.Loop(fixed_range_us)
    seq_gen.TimerWaitAndRestart(us_tick_delay)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.LoopEnd()
    
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=2)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=6)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=8)
    seq_gen.GetPulseCounterResult(MeasurementChannel.US_LUM, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=12)
    seq_gen.GetPulseCounterResult(MeasurementChannel.US_LUM, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=14)
    
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=17)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=18)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=19)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=20)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=21)

    seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset, ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset, ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 22)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def pmt_multi_range_measurement(window_ms=1.0, iterations=100):
    
    op_id = 'pmt_multi_range_measurement'
    meas_unit.ClearOperations()
    await load_pmt_multi_range_measurement(op_id, window_ms, iterations)
        
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    pmt1_cnt = results[0] + (results[1] << 32)
    pmt1_dt  = results[2] + (results[3] << 32)
    pmt1_al  = results[4]
    pmt1_ah  = results[5]
    pmt2_cnt = results[6] + (results[7] << 32)
    pmt2_dt  = results[8] + (results[9] << 32)
    pmt2_al  = results[10]
    pmt2_ah  = results[11]

    results = {}

    results['pmt1_cnt_mean'] = pmt1_cnt / iterations
    results['pmt1_dt_mean'] = pmt1_dt / iterations
    results['pmt1_al_mean'] = pmt1_al / iterations
    results['pmt1_ah_mean'] = pmt1_ah / iterations
    results['pmt2_cnt_mean'] = pmt2_cnt / iterations
    results['pmt2_dt_mean'] = pmt2_dt / iterations
    results['pmt2_al_mean'] = pmt2_al / iterations
    results['pmt2_ah_mean'] = pmt2_ah / iterations

    return results


async def load_pmt_multi_range_measurement(op_id, window_ms=1.0, window_count=100):
    if (window_ms < 0.001):
        raise ValueError(f"window_ms must be greater or equal to 0.001 ms")
    if (window_count < 1) or (window_count > 65536):
        raise ValueError(f"window_count must be in the range [1, 65536]")

    fixed_range_us = 20
    auto_range_us = round(window_ms * 1000) - fixed_range_us
    auto_range_coarse, auto_range_fine = divmod(auto_range_us, 65536)

    full_reset_delay = 40000    # 400 us
    reset_low_delay = 1000      #  10 us
    reset_off_delay = 1000      #  10 us
    switch_delay = 25           # 250 ns
    sample_offset_delay = 2     #  20 ns
    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # pmt1_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # pmt1_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt1_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # pmt1_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # pmt1_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 6)  # pmt2_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 7)  # pmt2_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 8)  # pmt2_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 9)  # pmt2_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt2_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt2_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # pmt3_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # pmt3_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # pmt3_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # pmt3_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # ref_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # ref_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=18)  # abs_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=19)  # abs_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=20)  # aux_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=21)  # aux_ah

    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.Loop(window_count)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset, ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset, ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.TimerWaitAndRestart(reset_low_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range, pmt2=IntegratorMode.integrate_in_high_range)

    seq_gen.TimerWaitAndRestart(reset_off_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_low_range, pmt2=IntegratorMode.integrate_in_high_range)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.low_range_reset, abs=IntegratorMode.low_range_reset, aux=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(sample_offset_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2 | TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_autorange, abs=IntegratorMode.integrate_autorange, aux=IntegratorMode.integrate_autorange)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if auto_range_coarse > 0:
        seq_gen.Loop(auto_range_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if auto_range_fine > 0:
        seq_gen.Loop(auto_range_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset, ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_with_fixed_range, abs=IntegratorMode.integrate_with_fixed_range, aux=IntegratorMode.integrate_with_fixed_range)
    
    seq_gen.Loop(fixed_range_us)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.LoopEnd()
    
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2 | TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)
    
    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=True, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=True, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=True, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=True, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=11)
    
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=2)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=6)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=8)
    seq_gen.GetPulseCounterResult(MeasurementChannel.US_LUM, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=12)
    seq_gen.GetPulseCounterResult(MeasurementChannel.US_LUM, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=14)

    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=17)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=18)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=19)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=20)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=21)

    seq_gen.LoopEnd()

    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset, ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset, ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 22)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def set_led_current(current, source='smu', channel='led1', led_type='red'):
    if source.startswith('smu'):
        return await smu_set_led_current(current, source, channel, led_type)
    if source.startswith('fmb'):
        return await fmb_set_led_current(current, source, channel, led_type)
    
    raise Exception(f"Invalid source: {source}")


led_compliance_voltage = {'red':2.0, 'green':2.9, 'blue':3.1}

async def smu_set_led_current(current=1, source='smu', channel='led1', led_type='green'):
    if not source.startswith('smu'):
        raise ValueError(f"Invalid source: {source}")
    
    from pymeasure.instruments.keithley import Keithley2450

    led_color = led_type.split('_')[0]

    # VISA address: USB[board]::manufacturer ID::model code::serial number[::USB interface number][::INSTR]
    smu = Keithley2450('USB0::1510::9296::?*::INSTR')
    smu.apply_current()
    smu.measure_voltage()
    smu.compliance_voltage = led_compliance_voltage[led_color]
    current = round(current)
    if (current > 0) and (current <= 30000):
        smu.source_current = current * 1e-6
        smu.enable_source()
    else:
        smu.shutdown()

    await asyncio.sleep(0.2)
    return current


#TODO New led names: r1, r2, g1, g2, ... and use different driver channels for the adjustment.
fmb_led_channel = {
    'led1' : {'bc':FMBAnalogOutput.BCG, 'gs':FMBAnalogOutput.GSG1},
    'led2' : {'bc':FMBAnalogOutput.BCR, 'gs':FMBAnalogOutput.GSR3},
    'led3' : {'bc':FMBAnalogOutput.BCG, 'gs':FMBAnalogOutput.GSG2},
    'led4' : {'bc':FMBAnalogOutput.BCB, 'gs':FMBAnalogOutput.GSB3},
}

fmb_full_scale_current = {
    'fmb'    :   127,   # Brightness Value [0, 127]
    'fmb#7'  :  2129,   # Measured with Keithley Multimeter (FMB#7,  24k3, Rev 2)
    'fmb#12' :  2084,   # Measured with Keithley Multimeter (FMB#12, 24k3, Rev 2)
    'fmb#15' : 11084,   # Measured with Keithley Multimeter (FMB#15,  4k7, Rev 2)
}

async def fmb_set_led_current(current=17, source='fmb', channel='led1', led_type='green'):
    bc = fmb_led_channel[channel]['bc']
    gs = fmb_led_channel[channel]['gs']
    full_scale_current = fmb_full_scale_current[source]
    brightness = current / full_scale_current
    current = round(min(int(brightness * 128), 127) / 127 * full_scale_current)
    if (current > 0):
        await fmb_endpoint.SetAnalogOutput(bc, brightness)
        await fmb_endpoint.SetAnalogOutput(gs, 1.0)
    else:
        await fmb_endpoint.SetAnalogOutput(bc, 0.0)
        await fmb_endpoint.SetAnalogOutput(gs, 0.0)

    await asyncio.sleep(0.2)
    return current

